/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: TwitterContent.java,v 1.19 2006/10/10 07:47:02 mseaton Exp $$ */
package org.lamsfoundation.lams.tool.twitter;

import java.io.Serializable;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.lamsfoundation.lams.contentrepository.ItemNotFoundException;
import org.lamsfoundation.lams.contentrepository.NodeKey;
import org.lamsfoundation.lams.contentrepository.RepositoryCheckedException;
import org.lamsfoundation.lams.contentrepository.client.IToolContentHandler;


/**
 * <p>Persistent twitter object/bean that defines the content for the twitter tool.
 * Provides accessors and mutators to get/set twitter attributes</p>
 * @hibernate.class table="tl_fptwit10_content"
 * @author mtruong
 */
public class TwitterContent implements Serializable{
	
	/** identifier field */
    	private Long uid;
    
    	/** non-nullable persistent field */
	private Long twitterContentId;
	
	/** nullable persistent field */
	private String title;
	
	/** nullable persistent field */
	private String content;
	
	/** nullable persistent field */
	private String onlineInstructions;
	
	/** nullable persistent field */
	private String offlineInstructions;
	
	/** nullable persistent field */
	private String teacherTwitterUsername;
	
	/** nullable persistent field */
	private String learnersTwitterUsernames;
	
	/** nullable persistent field */
	private String hashtags;
	
	/** nullable persistent field */
	private int tweetNumber;
	
	/** nullable persistent field */
	private boolean defineLater;
	
	/** nullable persistent field */
	private boolean forceOffline;

	private Boolean reflectOnActivity;
	
	private String reflectInstructions;
	
	/** nullable persistent field */
	private boolean contentInUse;
	
	/** nullable persistent field */
	private Long creatorUserId;
	
	/** nullable persistent field */
	private Date dateCreated;
	
	/** nullable persistent field */
	private Date dateUpdated;
	
	/** nullable persistent field */
	private String activityHashtag = "";
	
	/** persistent field */
	private Set twitterSessions = new HashSet();
	
	private Set twitterAttachments = new HashSet();
	
	/** default constructor */
	public TwitterContent(){}
	
	/** minimal constructor (for tweet saving purposes) */
	public TwitterContent(Long uid){
	    this.uid = uid;
	}
	
	/** full constructor */
	public TwitterContent(Long twitterContentId,
							  String title,
							  String content,
							  String onlineInstructions,
							  String offlineInstructions,
							  String teacherTwitterUsername,
							  String learnersTwitterUsernames,
							  String hashtags,
							  int tweetNumber,
							  boolean defineLater,
							  boolean forceOffline,
							  boolean reflectOnActivity,
							  String reflectInstructions,
							  boolean contentInUse,
							  Long creatorUserId,
							  Date dateCreated,
							  Date dateUpdated)
	{
		this.twitterContentId = twitterContentId;
		this.title = title;
		this.content = content;
		this.onlineInstructions = onlineInstructions;
		this.offlineInstructions = offlineInstructions;
		this.teacherTwitterUsername = teacherTwitterUsername;
		this.learnersTwitterUsernames = learnersTwitterUsernames;
		this.hashtags = hashtags;
		this.tweetNumber = tweetNumber;
		this.defineLater = defineLater;
		this.forceOffline = forceOffline;
		this.reflectOnActivity = reflectOnActivity;
		this.reflectInstructions = reflectInstructions;
		this.contentInUse = contentInUse;
		this.creatorUserId = creatorUserId;
		this.dateCreated = dateCreated;
		this.dateUpdated = dateUpdated;
	}
	
	/**
	 * Minimal Constructor used to initialise values for the TwitterContent object
	 * @return
	 */
	
	public TwitterContent(Long twitterContentId,
	        				  String title,
	        				  String content,
	        				  String onlineInstructions,
	        				  String offlineInstructions,
	        				  String teacherTwitterUsername,
	        				  String learnersTwitterUsernames,
	        				  String hashtags,
	        				  int tweetNumber,
	        				  Date dateCreated)
	{
	    this.twitterContentId = twitterContentId;
		this.title = title;
		this.content = content;
		this.onlineInstructions = onlineInstructions;
		this.offlineInstructions = offlineInstructions;
		this.teacherTwitterUsername = teacherTwitterUsername;
		this.learnersTwitterUsernames = learnersTwitterUsernames;
		this.hashtags = hashtags;
		this.tweetNumber = tweetNumber;
		this.defineLater = false;
		this.forceOffline = false;
		this.reflectOnActivity = false;
		this.contentInUse = false;
		this.creatorUserId = null;
		this.dateCreated = dateCreated;
		this.dateUpdated = null;
	}
	
	
    
	/**
	 *		 @hibernate.property
     *       column="content"
     *       length="65535"
	 */
	
	public String getContent() {
		return content;
	}

	public void setContent(String content) {
		this.content = content;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="creator_user_id"
     *		length="20"
     */	
	public Long getCreatorUserId() {
		return creatorUserId;
	}
	
	public void setCreatorUserId(Long creatorUserId) {
		this.creatorUserId = creatorUserId;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="date_created"
     *		length="19"
     */
	public Date getDateCreated() {
		return dateCreated;
	}
	
	public void setDateCreated(Date dateCreated) {
		this.dateCreated = dateCreated;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="date_updated"
     *		length="19"
     */
	public Date getDateUpdated() {
		return dateUpdated;
	}
	
	public void setDateUpdated(Date dateUpdated) {
		this.dateUpdated = dateUpdated;
	}
	
	/** 
	 *		@hibernate.property
     *		column="define_later"
     *		length="1"
     */
	public boolean isDefineLater() {
		return defineLater;
	}
	
	public void setDefineLater(boolean defineLater) {
		this.defineLater = defineLater;
	}
	
	/** 
	 *		@hibernate.property
     *		column="force_offline"
     *		length="1"
     */
	public boolean isForceOffline() {
		return forceOffline;
	}
	
	public void setForceOffline(boolean forceOffline) {
		this.forceOffline = forceOffline;
	}
	
	/** 
	 *		@hibernate.property
     *		column="reflect_on_activity"
     *		length="1"
     */
	public boolean getReflectOnActivity() {
		return reflectOnActivity;
	}
	
	public void setReflectOnActivity(boolean reflectOnActivity) {
		this.reflectOnActivity = reflectOnActivity;
	}
	
	/**
	 *		 @hibernate.property
     *       column="reflect_instructions"
     *       length="65535"
	 */
	public String getReflectInstructions() {
		return reflectInstructions;
	}

	public void setReflectInstructions(String reflectInstructions) {
		this.reflectInstructions = reflectInstructions;
	}
	
	 /** 
	 *		@hibernate.property
     *		column="content_in_use"
     *		length="1"
     */
    
    public boolean isContentInUse() {
        return contentInUse;
    }
    /**
     * @param contentInUse The contentInUse to set.
     */
    public void setContentInUse(boolean contentInUse) {
        this.contentInUse = contentInUse;
    }
	
	/** 
	 *		@hibernate.property
     *      column="twitter_content_id"
     *      length="20"
     *      not-null="true"
     */
	
	public Long getTwitterContentId() {
		return twitterContentId;
	}
	
	public void setTwitterContentId(Long twitterContentId) {
		this.twitterContentId = twitterContentId;
	}
	/**
	 * 		@hibernate.set
     *      lazy="true"
     *      inverse="true"
     *      cascade="all-delete-orphan"
     *     	@hibernate.collection-key
     *      column="twitter_content_uid"
     *     	@hibernate.collection-one-to-many
     *      class="org.lamsfoundation.lams.tool.twitter.TwitterSession"
     */
	public Set getTwitterSessions() {
		if (this.twitterSessions == null)
		{
			setTwitterSessions(new HashSet());
		}
		return twitterSessions;
	}
	
	public void setTwitterSessions(Set twitterSessions) {
		this.twitterSessions = twitterSessions;
	}
	
	/**
     *  	@hibernate.set
     *      lazy="true"
     *      inverse="true"
     *      cascade="all-delete-orphan"
     *      @hibernate.collection-key
     *      column="twitter_content_uid"
     * 		@hibernate.collection-one-to-many
     *      class="org.lamsfoundation.lams.tool.twitter.TwitterAttachment"
     */
    public Set getTwitterAttachments() {
        return twitterAttachments;
    }
    /**
     * @param twitterAttachments The twitterAttachments to set.
     */
    public void setTwitterAttachments(Set twitterAttachments) {
        this.twitterAttachments = twitterAttachments;
    }
	
	
	
	/**
	 * 
	 *		@hibernate.property
     *		column="offline_instructions"
     *		length="65535"
     */	
	public String getOfflineInstructions() {
		return offlineInstructions;
	}
	
	public void setOfflineInstructions(String offlineInstructions) {
		this.offlineInstructions = offlineInstructions;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="teacherTwitterUsername"
     *		length="255"
     */	
	public String getTeacherTwitterUsername() {
		return teacherTwitterUsername;
	}
	
	public void setTeacherTwitterUsername(String teacherTwitterUsername) {
		this.teacherTwitterUsername = teacherTwitterUsername;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="learnersTwitterUsernames"
     *		length="511"
     */	
	public String getLearnersTwitterUsernames() {
		return learnersTwitterUsernames;
	}
	
	public void setLearnersTwitterUsernames(String learnersTwitterUsernames) {
		this.learnersTwitterUsernames = learnersTwitterUsernames;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="hashtags"
     *		length="255"
     */	
	public String getHashtags() {
		return hashtags;
	}
	
	public void setHashtags(String hashtags) {
		this.hashtags = hashtags;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="tweetNumber"
     *		length="2"
     */	
	public int getTweetNumber() {
		return tweetNumber;
	}
	
	public void setTweetNumber(int tweetNumber) {
		this.tweetNumber = tweetNumber;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="activity_hashtag"
     *		length="12"
     */	
	public String getActivityHashtag() {
		return activityHashtag;
	}
	
	public void setActivityHashtag(String activityHashtag) {
		this.activityHashtag = activityHashtag;
	}
	
	/**
	 * 
	 *		@hibernate.property
     *		column="online_instructions"
     *		length="65535"
     */	
	public String getOnlineInstructions() {
		return onlineInstructions;
	}
	
	public void setOnlineInstructions(String onlineInstructions) {
		this.onlineInstructions = onlineInstructions;
	}
	
	/**
	 * 		@hibernate.property
     *      column="title"
     *      length="65535"
	 */
	public String getTitle() {
		return title;
	}
	
	public void setTitle(String title) {
		this.title = title;
	}
	
	 /**
      * 	@hibernate.id
      *     generator-class="native"
      *     type="java.lang.Long"
      *     column="uid"
      *     unsaved-value="0"
      */
    public Long getUid() {
        return uid;
    }
    
    public void setUid(Long uid) {
        this.uid = uid;
    }
	

   
    
    
    
    
	/** 
	 * Creates a new TwitterContent object from the supplied object.
	 * Assigns it the toContendId. Also copies all the items in the attachment set
	 * to the new object's attachment set. So while the two contents have different
	 * attachment records, they point to the same entries in the database.
	 * 
	 * @param twitter			TwitterContent object containing the content to copy from
	 * @param toContentId 	The new Id of the new twitter object
	 * @return newContent	The new twitter content object
	 * @throws RepositoryCheckedException 
	 * @throws ItemNotFoundException 
	 */
	public static TwitterContent newInstance(TwitterContent twitter, Long toContentId, IToolContentHandler toolContentHandler) throws ItemNotFoundException, RepositoryCheckedException
	{
		TwitterContent newContent = new TwitterContent(toContentId,
														twitter.getTitle(),
														twitter.getContent(),
														twitter.getOnlineInstructions(),
														twitter.getOfflineInstructions(),
														twitter.getTeacherTwitterUsername(),
														twitter.getLearnersTwitterUsernames(),
														twitter.getHashtags(),
														twitter.getTweetNumber(),
														twitter.isDefineLater(),
														twitter.isForceOffline(),
														twitter.getReflectOnActivity(),
														twitter.getReflectInstructions(),
														twitter.isContentInUse(),
														twitter.getCreatorUserId(),
														twitter.getDateCreated(),
														twitter.getDateUpdated());
		
		if ( twitter.getTwitterAttachments() != null && twitter.getTwitterAttachments().size() > 0 ) {
			HashSet newAttachmentSet = new HashSet();
			Iterator iter = twitter.getTwitterAttachments().iterator();
			while (iter.hasNext()) {
				TwitterAttachment element = (TwitterAttachment) iter.next();
				TwitterAttachment newAttachment = new TwitterAttachment(newContent, element.getFilename(), element.isOnlineFile());
				//keep old value do not duplicate file
				newAttachment.setUuid(element.getUuid());
				newAttachment.setVersionId(element.getVersionId());
				
				newAttachmentSet.add(newAttachment);
			} 
			newContent.setTwitterAttachments(newAttachmentSet);
		}
		
		return newContent;
	}
   
	
 
}
