/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: ITwitterAttachmentDAO.java,v 1.6 2006/09/17 06:26:26 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.twitter.dao;

import java.util.List;

import org.lamsfoundation.lams.tool.twitter.TwitterAttachment;
import org.lamsfoundation.lams.tool.twitter.TwitterContent;
/**
 * @author mtruong
 *
 * <p>Handles database access for uploading, updates and removal of files.</p>
 */
public interface ITwitterAttachmentDAO {

    /**
     * Return the persistent instance of TwitterAttachment with the given
     * identifier <code>attachmentId</code>
     * 
     * @param attachmentId The unique identifier for attachment. If null, a null object will be returned.
     * @return an instance of TwitterAttachment
     */
    public TwitterAttachment retrieveAttachment(Long attachmentId);
    
    /**
     * Return the persistent instance of TwitterAttachment with the given
     * unique identifier <code>uuid</code>. 
     * 
     * @param uuid The uuid of the file, which is also the uuid of the file which is stored in the content repository.
     * @return an instance of TwitterAttachment
     */
    public TwitterAttachment retrieveAttachmentByUuid(Long uuid);
    
    /**
     * Return the persistent instance of TwitterAttachment with the given
     * <code>filename</code>. If there is more than one file with the same filename
     * then only the first file in the list will be returned.
     * 
     * @param filename The name of the file
     * @return an instance of TwitterAttachment
     */
    public TwitterAttachment retrieveAttachmentByFilename(String filename);
    
    /**
     * Returns a list of attachment ids which are associated with this
     * instance of TwitterContent. 
     * <p>For example, if the given instance <code>twitterContent</code> has a tool content id
     * of 3, and consider an extract of the tl_fptwit10_attachment table:</p>
     * <pre> 
     * 		 ----------------------------
     * 		 attachmentId | toolContentId
     * 		 ----------------------------
     * 			1		  | 	1
     * 			2		  | 	3
     * 			3		  | 	3
     * 			4 		  | 	1
     * 		 ----------------------------
     * </pre>
     * Then a call to <code>getAttachmentIdsFromContent</code> will return a list containing the values
     * 2 and 3. 
     * @param twitterContent
     * @return
     */
    public List getAttachmentIdsFromContent(TwitterContent twitterContent);
    
    /**
	 * <p>Persist the given persistent instance of TwitterAttachment.</p>
	 * 
	 * @param attachment The instance of TwitterAttachment to persist.
	 */
    public void saveAttachment(TwitterAttachment attachment);
    
    /**
	 * <p>Delete the given persistent instance of TwitterAttachment.</p>
	 * 
	 * @param attachment The instance of TwitterAttachment to delete.
	 */
    public void removeAttachment(TwitterAttachment attachment);
    
    /**
     * <p>Delete the given instance of TwitterAttachment with the
     * given <code>uuid</code>
     * 
     * @param uuid The unique id of the file/attachment.
     */
    public void removeAttachment(Long uuid);
    
}
