/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: TwitterContentDAO.java,v 1.8 2006/09/17 06:26:26 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.twitter.dao.hibernate;

import java.util.List;
import java.lang.Long;

import org.hibernate.FlushMode;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.lamsfoundation.lams.tool.twitter.TwitterContent;
import org.lamsfoundation.lams.tool.twitter.TwitterSession;
import org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO;
import org.springframework.orm.hibernate3.HibernateCallback;
import org.springframework.orm.hibernate3.HibernateTemplate;
import org.springframework.orm.hibernate3.support.HibernateDaoSupport;

/**
 * @author mtruong
 * <p>Hibernate implementation for database access to Twitter content for the twitter tool.</p>
 */

public class TwitterContentDAO extends HibernateDaoSupport implements ITwitterContentDAO {
	
	private static final String FIND_NB_CONTENT = "from " + TwitterContent.class.getName() + " as twitter where twitter.twitterContentId=?";
	
	
	private static final String LOAD_NB_BY_SESSION = "select twitter from TwitterContent twitter left join fetch "
        + "twitter.twitterSessions session where session.twitterSessionId=:sessionId";

	
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#getTwitterContentByUID(java.lang.Long) */
	public TwitterContent getTwitterContentByUID(Long uid)
	{
		 return (TwitterContent) this.getHibernateTemplate()
         .get(TwitterContent.class, uid);
	}
	
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#findTwitterContentById(java.lang.Long) */
	public TwitterContent findTwitterContentById(Long twitterContentId)
	{
	    String query = "from TwitterContent as twitter where twitter.twitterContentId = ?";
		List content = getHibernateTemplate().find(query,twitterContentId);
			
		if(content!=null && content.size() == 0)
		{			
			return null;
		}
		else
		{
			return (TwitterContent)content.get(0);
		}
	
	}
    	
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#getTwitterContentBySession(java.lang.Long) */
	public TwitterContent getTwitterContentBySession(final Long twitterSessionId)
	{
		 return (TwitterContent) getHibernateTemplate().execute(new HibernateCallback()
                {

                    public Object doInHibernate(Session session) throws HibernateException
                    {
                        return session.createQuery(LOAD_NB_BY_SESSION)
                                      .setLong("sessionId",
                                      		twitterSessionId.longValue())
                                      .uniqueResult();
                    }
                });
	}
	
	
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#saveTwitterContent(org.lamsfoundation.lams.tool.twitter.TwitterContent) */
	public void saveTwitterContent(TwitterContent twitterContent)
    {
    	this.getHibernateTemplate().save(twitterContent);
    }
    
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#updateTwitterContent(org.lamsfoundation.lams.tool.twitter.TwitterContent) */
	public void updateTwitterContent(TwitterContent twitterContent)
    {
    	this.getHibernateTemplate().update(twitterContent);
    }

   
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#removeTwitter(java.lang.Long)*/
	public void removeTwitter(Long twitterContentId)
    {       
       	if ( twitterContentId != null) {
			//String query = "from org.lamsfoundation.lams.tool.twitter.TwitterContent as twitter where twitter.twitterContentId=?";
			List list = getSession().createQuery(FIND_NB_CONTENT)
				.setLong(0,twitterContentId.longValue())
				.list();
			
			if(list != null && list.size() > 0){
				TwitterContent twitter = (TwitterContent) list.get(0);
				this.getSession().setFlushMode(FlushMode.AUTO);
				this.getHibernateTemplate().delete(twitter);
				this.getHibernateTemplate().flush();
			}
		}
    }
    
	/** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#removeTwitter(org.lamsfoundation.lams.tool.twitter.TwitterContent)*/
    public void removeTwitter(TwitterContent twitterContent)
    {
    	removeTwitter(twitterContent.getTwitterContentId());
    }
   
    /** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#removeTwitterSessions(org.lamsfoundation.lams.tool.twitter.TwitterContent)*/
    public void removeTwitterSessions(TwitterContent twitterContent)
    {
    	this.getHibernateTemplate().deleteAll(twitterContent.getTwitterSessions());
    }
    
    /** @see org.lamsfoundation.lams.tool.twitter.dao.ITwitterContentDAO#addTwitterSession(java.lang.Long, org.lamsfoundation.lams.tool.twitter.TwitterSession) */
    public void addTwitterSession(Long twitterContentId, TwitterSession twitterSession)
    {
        TwitterContent content = findTwitterContentById(twitterContentId);
        twitterSession.setTwitterContent(content);
        content.getTwitterSessions().add(twitterSession);
        this.getHibernateTemplate().saveOrUpdate(twitterSession);
        this.getHibernateTemplate().saveOrUpdate(content);        
    }
  
}
