/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: ITwitterContentDAO.java,v 1.6 2006/09/17 06:26:26 fmalikoff Exp $$ */
package org.lamsfoundation.lams.tool.twitter.dao;

import org.lamsfoundation.lams.tool.twitter.TwitterContent;
import org.lamsfoundation.lams.tool.twitter.TwitterSession;

/**
 * <p>Interface for the TwitterContent DAO, defines methods needed to access/modify
 * twitter content</p>
 * @author mtruong
 */
public interface ITwitterContentDAO {
    
    
    /**
	 * <p>Return the persistent instance of a TwitterContent  
	 * with the given identifier <code>uid</code>, returns null if not found. </p>
	 * 
	 * @param uid an identifier for the TwitterContent instance.
	 * @return the persistent instance of a TwitterContent or null if not found
	 */
	public TwitterContent getTwitterContentByUID(Long uid);
	
	/**
	 * <p> Return the persistent instance of a TwitterContent
	 * with the given tool content id <code>twitterContentId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param twitterContentId The tool content id
	 * @return the persistent instance of a TwitterContent or null if not found.
	 */
	public TwitterContent findTwitterContentById(Long twitterContentId);
	
	/**
     * <p> Returns the persistent instance of TwitterContent
     * with the given tool session id <code>twitterSessionId</code>, returns null if not found.
     * 
     * @param twitterSessionId The tool session id
     * @return a persistent instance of TwitterContent or null if not found.
     */
	public TwitterContent getTwitterContentBySession(Long twitterSessionId);
    
	/**
	 * <p>Persist the given persistent instance of TwitterContent.</p>
	 * 
	 * @param twitterContent The instance of TwitterContent to persist.
	 */
    public void saveTwitterContent(TwitterContent twitterContent);
    
    /**
     * <p>Update the given persistent instance of TwitterContent.</p>
     * 
     * @param twitterContent The instance of TwitterContent to persist.
     */
    public void updateTwitterContent(TwitterContent twitterContent);

    /**
     * <p>Delete the given instance of TwitterContent</p>
     * 
     * @param twitterContent The instance of TwitterContent to delete. 
     */
    public void removeTwitter(TwitterContent twitterContent);
    
    /**
     * <p>Delete the given instance of TwitterContent with the
     * given tool content id <code>twitterContentId</code>
     * 
     * @param twitterContentId The tool content Id. 
     */
    public void removeTwitter(Long twitterContentId);
   
    
    /**
     * <p>Deletes all instances of TwitterSession that are associated
     * with the given instance of TwitterContent</p>
     * 
     * @param twitterContent The instance of TwitterContent in which corresponding instances of TwitterSession should be deleted.
     */
    public void removeTwitterSessions(TwitterContent twitterContent);
    
    /**
     * <p>Creates a persistent instance of TwitterSession which is associated
     * with the TwitterContent with tool content id <code>twitterContentId</code> 
     * </p>
     * 
     * @param twitterContentId The tool content id
     * @param twitterSession The instance of TwitterSession to add
     */
    public void addTwitterSession(Long twitterContentId, TwitterSession twitterSession);
    
  }
