/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: ITwitterService.java,v 1.19 2006/11/24 00:28:44 asukkar Exp $$ */
package org.lamsfoundation.lams.tool.twitter.service;

import java.io.InputStream;
import java.util.List;

import org.lamsfoundation.lams.contentrepository.NodeKey;
import org.lamsfoundation.lams.contentrepository.RepositoryCheckedException;
import org.lamsfoundation.lams.notebook.model.NotebookEntry;
import org.lamsfoundation.lams.tool.twitter.TwitterAttachment;
import org.lamsfoundation.lams.tool.twitter.TwitterContent;
import org.lamsfoundation.lams.tool.twitter.TwitterSession;
import org.lamsfoundation.lams.tool.twitter.TwitterUser;
import org.lamsfoundation.lams.tool.twitter.TwitterUserTweet;


/**
 * Defines the contract that the tool service provider must follow
 * 
 * @author mtruong
 */
public interface ITwitterService{
	
    //===================================================================
    // TwitterContent access methods
    //===================================================================	
    
  
    
    /**
     * <p> Returns the persistent instance of TwitterContent
     * with the given tool session id <code>twitterSessionId</code>, returns null if not found.
     * 
     * @param twitterSessionId The tool session id
     * @return a persistent instance of TwitterContent or null if not found.
     */	
	public TwitterContent retrieveTwitterBySessionID(Long twitterSessionId);
    
    /**
	 * <p>Retrieve an instance of TwitterContent with the given
     * tool content id <code>twitterContentId</code> </p>
     * @param twitterContentId The tool content id
     * @return an instance of TwitterContent
     */
	public TwitterContent retrieveTwitter(Long twitterContentId);
	
	
	/**
	 * <p>Persist/Update the given persistent instance of TwitterContent.</p>
	 * 
	 * @param twitterContent The instance of TwitterContent to persist.
	 */
	public void saveTwitter(TwitterContent twitterContent);
	
	
	 /**
     * <p>Deletes all instances of TwitterSession that are associated
     * with the given instance of TwitterContent</p>
     * 
     * @param twitterContent The instance of TwitterContent in which corresponding instances of TwitterSession should be deleted.
     */
	public void removeTwitterSessionsFromContent(TwitterContent twitterContent);
	
	/**
     * <p>Delete the given instance of TwitterContent with the
     * given tool content id <code>twitterContentId</code>
     * 
     * @param twitterContentId The tool content Id. 
     */
	public void removeTwitter(Long twitterContentId);
	
	/**
     * <p>Delete the given instance of TwitterContent</p>
     * 
     * @param twitterContent The instance of TwitterContent to delete. 
     */
    public void removeTwitter(TwitterContent twitterContent);
	
	
    //===================================================================
    // TwitterSession access methods
    //===================================================================
    /**
	 * <p> Return the persistent instance of a TwitterSession
	 * with the given tool session id <code>twitterSessionId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param twitterSessionId The tool session id
	 * @return the persistent instance of a TwitterSession or null if not found.
	 */
	public TwitterSession retrieveTwitterSession(Long twitterSessionId);
	

	/**
	 * Persists the new TwitterSession object into the database.
	 * 
	 * @param twitterSession the TwitterSession object to persist
	 */
	public void saveTwitterSession(TwitterSession twitterSession);
	
	
	/**
	 * Updates the values of the twitter session.
	 * @param twitterSession
	 */
	public void updateTwitterSession(TwitterSession twitterSession);
	
	/**
	 * Remove the twitter session object with session id of
	 * that specified in the argument.
	 * 
	 * @param twitterSessionId The id of the requested twitter object
	 *
	 */	
	public void removeSession(Long twitterSessionId);
	
	 /**
     * <p>Delete the given instance of TwitterSession</p>
     * 
     * @param twitterSession The instance of TwitterSession to delete. 
     */
    public void removeSession(TwitterSession twitterSession);
    
     
    /**
     * <p>Deletes all instances of TwitterUser that are associated
     * with the given instance of TwitterSession</p>
     * 
     * @param twitterSession The instance of TwitterSession in which corresponding instances of TwitterUser should be deleted.
     */
	public void removeTwitterUsersFromSession(TwitterSession twitterSession);
	
	  /**
     * <p> Returns the persistent instance of TwitterSession
     * with the given twitter user id<code>userId</code>, returns null if not found.
     * 
     * @param userId The user id
     * @return a persistent instance of TwitterSession or null if not found.
     */	
	public TwitterSession retrieveTwitterSessionByUserID(Long userId);
    
    //===================================================================
    // TwitterUser access methods
    //===================================================================
    /**
	 * <p> Return the persistent instance of a TwitterUser
	 * with the given user id<code>twitterUserId</code>,
	 * returns null if not found.</p>
	 * 
	 * @param twitterUserId The user id of the instance of TwitterUser
	 * @return the persistent instance of a TwitterUser or null if not found.
	 */
	public TwitterUser retrieveTwitterUser(Long twitterUserId, Long toolSessionId);
	
	/**
	 * <p> Return the persistent instance of a TwitterUser
	 * who has the user id <code>userId</code> and tool session id
	 * <code>sessionId</code>
	 * returns null if not found.</p>
	 * 
	 * @param userId. The id of the learner
	 * @param sessionId. The tool session id to which this user belongs to.
	 * @return the persistent instance of a TwitterUser or null if not found.
	 */
	public TwitterUser retrieveTwitterUserBySession(Long userId, Long sessionId);
	
	/**
	 * Persists the new TwitterUser object into the database.
	 * 
	 * @param twitterUser the TwitterUser object to persist
	 */
	public void saveTwitterUser(TwitterUser twitterUser);
	
	
	/**
	 * Updates the values of the twitter user.
	 * @param twitterUser
	 */
	public void updateTwitterUser(TwitterUser twitterUser);
	
	/**
	 * Remove the twitter user object with user id of
	 * that specified in the argument.
	 * 
	 * @param twitterUserId The id of the requested twitter object
	 * @param toolSessionId The id of the twitterUser's associated twitterSession
	 *
	 */	
	public void removeUser(Long twitterUserId, Long toolSessionId);
	
	 /**
     * <p>Delete the given instance of TwitterUser</p>
     * 
     * @param twitterUser The instance of TwitterUser to delete. 
     */
    public void removeUser(TwitterUser twitterUser);
 
    /**
     * <p> Saves the instance of TwitterSession to the database. 
     * This instance is added to the collection of sessions from 
     * TwitterContent with tool content id <code>twitterContentId</code> </p>
     * 
     * @param twitterContentId The tool content Id
     * @param session The instance of TwitterSession to persist
     */
    public void addSession(Long twitterContentId, TwitterSession session);
    
    /**
     * <p>Saves the instance of TwitterUser to the database.
     * <code>twitterUser</code> is added to the collection of users from 
     * TwitterSession with tool session id <code>twitterSessionId</code> </p>
     * 
     * @param twitterSessionId The tool session id
     * @param user The instance of TwitterUser to persist
     */
    public void addUser(Long twitterSessionId, TwitterUser twitterUser);
    
    /**
     * <p>Retrieves a list of the session IDs from the given instance of TwitterContent</p> 
     * @param content 
     * @return list of session ids (Long)
     */
    public List getSessionIdsFromContent(TwitterContent content);
    
    /**
     * <p>Returns the number of users in this session</p>
     * @param session
     * @return The number of users in the session
     */
    public int getNumberOfUsersInSession(TwitterSession session);
    
    /**
     * <p>Finds the number of learners that have participated in this tool activity
     * with the given toolContentId. It finds all the toolSessionIds relating to this
     * toolContentId, and calculates the number of users in each tool session(group).
     * Returns the total number of users across all sessions </p>
     * @param toolContentId 
     * @return the total number of users for this tool activity
     */
    public int calculateTotalNumberOfUsers(Long toolContentId);
    
    //===================================================================
    // TwitterUserTweet access methods
    //===================================================================
    
    /**
	 * <p> Return the number of persistent tweets
	 * with the given tool content id <code>toolContentId</code>
	 * who has the user id <code>userId</code>
	 * returns 0 if no tweet is found.</p>
	 * 
	 * @param toolContentId. The tool content id of the current activity.
	 * @param userId. The id of the learner
	 * @return the number of persistent tweets with (with the given tool content id and user id) or 0 if not found.
	 */
    public int retrieveNumberOfTweetsByUser(Long toolContentId, Long userId);
    
    /**
     * <p> Return the persistent instances of the first four TwitterUserTweet
     * with the given tool content id <code>toolContentId</code>
     * returns null if not found.</p>
     * 
     * @param toolContentId. The tool content id of the current activity.
     * @return the persistent instances of the first four TwitterUserTweet or null if not found.
     */
    public TwitterUserTweet[] retrieveTwitterUserTweet(Long toolContentId);
	
    /**
     * <p> Return the persistent instances of the first four TwitterUserTweet
     * who has the user id <code>userId</code> and tool content id
     * <code>toolContentId</code>
     * returns null if not found.</p>
     * 
     * @param userId. The id of the learner
     * @param toolContentId. The tool content id of the current activity.
     * @return the persistent instances of the first four TwitterUserTweet or null if not found.
     */
    public TwitterUserTweet[] retrieveTwitterUserTweetByUser(Long userId, Long toolContentId);
	
    /**
     * Persists the new TwitterUserTweet object into the database.
     * 
     * @param twitterUserTweet the TwitterUserTweet object to persist
     */
    public void saveTwitterUserTweet(TwitterUserTweet twitterUserTweet);

    
    //===================================================================
    // TwitterAttachment access methods
    //===================================================================
    
    /**
	 * <p>Retrieve an instance of TwitterAttachment with the given
     * attachment id <code>attachmentId</code> </p>
     * @param attachmentId The id for the attachment file
     * @return an instance of TwitterAttachment
     */
    public TwitterAttachment retrieveAttachment(Long attachmentId);
    
    /**
	 * <p>Retrieve the file attachment with the given uuid </p>
     * @param uuid The unique identifier for the file, corresponds to the uuid for the file stored in content repository
     * @return an instance of TwitterAttachment
     */
    public TwitterAttachment retrieveAttachmentByUuid(Long uuid);
    
    /**
	 * <p>Retrieve an instance of TwitterAttachment with the 
     * filename <code>filename</code> </p>
     * @param filename The filename of the attachment that you want to retrieve
     * @return an instance of TwitterAttachment
     */
    public TwitterAttachment retrieveAttachmentByFilename(String filename);
    
    /**
     * <p>Retrieve the list of attachment ids with the given instance of TwitterContent</p>
     * @param twitterContent The given instance of TwitterContent
     * @return List. the list of attachment ids (java.lang.Long)
     */
    public List getAttachmentIdsFromContent(TwitterContent twitterContent);
    
    /**
     * <p> Saves (persists) or update the TwitterAttachment object in the
     * database.</p>
     * @param content The overall twitter content object to which the attachment is to be added
     * @param attachment The instance of TwitterAttachment to save
     */
    public void saveAttachment(TwitterContent content, TwitterAttachment attachment);
    
    /**
     * Removes the TwitterAttachment object from the database.
     * @param content The overall twitter content object to which the attachment is to be added
     * @param attachment The instance of TwitterAttachment to delete.
     */
    public void removeAttachment(TwitterContent content, TwitterAttachment attachment) throws RepositoryCheckedException;
    
	/** 
	 * Add a file to the content repository. Does not add a record to the twitter tables.
	 * @throws RepositoryCheckedException 
	 */
	public NodeKey uploadFile(InputStream istream, String filename, String contentType, String fileType) throws RepositoryCheckedException;

    /**
     * This method retrieves the default content id.
     * @param toolSignature The tool signature which is defined in lams_tool table.
     * @return the default content id
     */
    public Long getToolDefaultContentIdBySignature(String toolSignature);
    
    public Long createNotebookEntry(Long id, Integer idType, String signature, Integer userID, String entry);
    
    public NotebookEntry getEntry(Long id, Integer idType, String signature, Integer userID);
    
	public void updateEntry(NotebookEntry notebookEntry);
    
    public List getUsersBySession(Long sessionId);
}
