/****************************************************************
 * Copyright (C) 2005 LAMS Foundation (http://lamsfoundation.org)
 * =============================================================
 * License Information: http://lamsfoundation.org/licensing/lams/2.0/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2.0 
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 * 
 * http://www.gnu.org/licenses/gpl.txt
 * ****************************************************************
 */

/* $$Id: TwitterDataAccessTestCase.java,v 1.15 2008/09/03 07:40:45 asukkar Exp $$ */
package org.lamsfoundation.lams.tool.twitter;

import org.lamsfoundation.lams.test.AbstractLamsTestCase;
import org.lamsfoundation.lams.tool.twitter.dao.hibernate.TwitterContentDAO;
import org.lamsfoundation.lams.tool.twitter.dao.hibernate.TwitterSessionDAO;
import org.lamsfoundation.lams.tool.twitter.dao.hibernate.TwitterUserDAO;
import org.lamsfoundation.lams.tool.twitter.dao.hibernate.TwitterAttachmentDAO;
import org.lamsfoundation.lams.tool.twitter.TwitterContent;
import java.util.Date;
import org.lamsfoundation.lams.tool.twitter.TwitterConstants;

/**
 * @author mtruong
 */
public class TwitterDataAccessTestCase extends AbstractLamsTestCase
{

    //---------------------------------------------------------------------
    // DAO instances for initializing data
    //---------------------------------------------------------------------
    protected TwitterContentDAO twitterDAO;
    protected TwitterSessionDAO twitterSessionDAO;
    protected TwitterUserDAO twitterUserDAO;
    protected TwitterAttachmentDAO attachmentDAO;
   
    //---------------------------------------------------------------------
    // Domain Object instances
    //---------------------------------------------------------------------
    protected TwitterContent twitterContent;
    protected TwitterSession twitterSession;
    protected TwitterUser twitterUser;
    protected TwitterAttachment twitterAttachment;

    
    //---------------------------------------------------------------------
    // DATA USED FOR TESTING PURPOSES ONLY
    //---------------------------------------------------------------------
    
    protected final long ONE_DAY = 60 * 60 * 1000 * 24;
    
    protected final Long TEST_NB_ID = new Long(1500);
    protected final Long TEST_COPYNB_ID = new Long(3500);
    
    protected final String TEST_TITLE = "Test Title";
    protected final String TEST_CONTENT = "Welcome! We hope you enjoy the activities that are set out.";
	protected final String TEST_ONLINE_INSTRUCTIONS = "Put your online instructions here";
	protected final String TEST_OFFLINE_INSTRUCTIONS = "Put your offline instructions here";
	protected final boolean TEST_DEFINE_LATER = false;
	protected final boolean TEST_REFLECT_ON_ACTIVITY = false;
	protected final String TEST_REFLECT_INSTRUCTIONS = "Put your reflections instructions here";
	protected final boolean TEST_FORCE_OFFLINE = false;
	protected final boolean TEST_CONTENT_IN_USE = false;
	protected final Date TEST_DATE_CREATED = new Date(System.currentTimeMillis());
	protected final Date TEST_DATE_UPDATED = new Date();
	protected final Long TEST_CREATOR_USER_ID = new Long(1300);
	
	protected final Long TEST_SESSION_ID = new Long(1400);
	protected final String TEST_SESSION_NAME = "Session 1400";
	protected final Date TEST_SESSION_START_DATE = new Date(System.currentTimeMillis());
	protected final Date TEST_SESSION_END_DATE = new Date(System.currentTimeMillis() + ONE_DAY);
	protected final String TEST_SESSION_STATUS = TwitterSession.NOT_ATTEMPTED;
	
	protected final Long TEST_USER_ID = new Long(1600);
	protected final String TEST_USERNAME = "testUsername";
	protected final String TEST_FULLNAME = "Test User Fullname";
	protected final String TEST_USER_STATUS = TwitterUser.INCOMPLETE;
	
	protected final String TEST_FILENAME = "testFilename";
	protected final boolean TEST_IS_ONLINE_FILE = true;
	protected final Long TEST_UUID =  new Long(2002);
	    
	
	//---------------------------------------------------------------------
    // DEFAULT DATA INSERTED BY BUILD-DB ANT TASK
    //---------------------------------------------------------------------
	protected final Long DEFAULT_CONTENT_ID = TwitterConstants.DEFAULT_CONTENT_ID;
	protected final String DEFAULT_TITLE = "Welcome";
	protected final String DEFAULT_CONTENT = "Welcome to these activities";
	protected final String DEFAULT_ONLINE_INSTRUCTIONS = "Enter the online instructions here";
	protected final String DEFAULT_OFFLINE_INSTRUCTIONS = "Enter the offline instructions here";
	protected final boolean DEFAULT_DEFINE_LATER = false;
	protected final boolean DEFAULT_FORCE_OFFLINE = false;
	protected final boolean DEFAULT_CONTENT_IN_USE = false;
	protected final Long DEFAULT_CREATOR_USER_ID = TwitterConstants.DEFAULT_CREATOR_ID;
	protected final Long DEFAULT_SESSION_ID = TwitterConstants.DEFAULT_SESSION_ID;
	protected final String DEFAULT_SESSION_STATUS = TwitterSession.INCOMPLETE;
	protected final Long DEFAULT_USER_ID = new Long(2600);
	protected final String DEFAULT_USERNAME = "test";
	protected final String DEFAULT_FULLNAME = "test";
	protected final String DEFAULT_USER_STATUS = TwitterUser.INCOMPLETE;
	
	
	
	/** Default Constructor */
	public TwitterDataAccessTestCase(String name)
	{
		super(name);
	}
	
	//---------------------------------------------------------------------
    // Inherited Methods
    //---------------------------------------------------------------------
	
    protected void setUp() throws Exception {
    	super.setUp();
        twitterDAO = (TwitterContentDAO) this.context.getBean("twitterContentDAO");
        twitterSessionDAO = (TwitterSessionDAO) this.context.getBean("twitterSessionDAO");
        twitterUserDAO = (TwitterUserDAO) this.context.getBean("twitterUserDAO");
        attachmentDAO = (TwitterAttachmentDAO)this.context.getBean("twitterAttachmentDAO");
    }

    protected void tearDown() throws Exception 
    {
    	super.tearDown();
    }

	 /** Define the context files. Overrides method in AbstractLamsTestCase */
    protected String[] getContextConfigLocation() {
    	return new String[] {
    	        "org/lamsfoundation/lams/localApplicationContext.xml",
    			"org/lamsfoundation/lams/lesson/lessonApplicationContext.xml",
    			"org/lamsfoundation/lams/toolApplicationContext.xml",
    			"org/lamsfoundation/lams/learning/learningApplicationContext.xml",
    			"org/lamsfoundation/lams/tool/twitter/testApplicationContext.xml"};
    }
    
    /** Define the sessionFactory bean name located in testApplication.xml. */
    protected String getHibernateSessionFactoryName()
    {
    	return "twitterSessionFactory";
    }
    
    protected void initTwitterContentData()
    {
    	  	twitterContent = new TwitterContent(TEST_NB_ID,	
												TEST_TITLE,
												TEST_CONTENT,
												TEST_ONLINE_INSTRUCTIONS,
												TEST_OFFLINE_INSTRUCTIONS,
												TEST_DEFINE_LATER,
												TEST_FORCE_OFFLINE,
												TEST_REFLECT_ON_ACTIVITY,
												TEST_REFLECT_INSTRUCTIONS,
												TEST_CONTENT_IN_USE,
												TEST_CREATOR_USER_ID,
												TEST_DATE_CREATED,
												TEST_DATE_UPDATED);
	    	
	    	twitterDAO.saveTwitterContent(twitterContent);
	 
	}
    
    protected void cleanTwitterContentData(Long contentId)
    {
    	twitterDAO.removeTwitter(contentId);
    	//it correspondingly removes all the sessions and users along with it.
    }
    
   
    protected Long getTestTwitterId()
    {
        return this.TEST_NB_ID;
    }
    
    protected void initTwitterSessionContent()
    {

    	TwitterContent twitter = twitterDAO.findTwitterContentById(TEST_NB_ID);
    	
    	twitterSession = new TwitterSession(TEST_SESSION_ID,
    									   TEST_SESSION_NAME,
    									   twitter,
										   TEST_SESSION_START_DATE,
										   TEST_SESSION_END_DATE,
										   TEST_SESSION_STATUS);
    	twitterSessionDAO.saveTwitterSession(twitterSession);
    	
    	//associate the session with the content
    	twitter.getTwitterSessions().add(twitterSession);
    	
    }
    
    protected void initTwitterUserData()
    {
        TwitterSession ns = twitterSessionDAO.findTwitterSessionById(TEST_SESSION_ID);
        
        TwitterUser user = new TwitterUser(TEST_USER_ID,
                									ns,
                									TEST_USERNAME,
                									TEST_FULLNAME,
                									TEST_USER_STATUS);
        
        twitterUserDAO.saveTwitterUser(user);
        
        ns.getTwitterUsers().add(user);
    }
    
    /* TODO: have to upload attachment to repository */
    protected void initTwitterAttachmentData()
    {
        TwitterAttachment attachment = new TwitterAttachment();
        TwitterContent twitter = twitterDAO.findTwitterContentById(TEST_NB_ID);
        
        attachment.setFilename(TEST_FILENAME);
        attachment.setOnlineFile(TEST_IS_ONLINE_FILE);
        attachment.setTwitterContent(twitterContent);
        attachment.setUuid(TEST_UUID);
	     
	    attachmentDAO.saveAttachment(attachment);
	    
	    twitter.getTwitterAttachments().add(attachment);        
    }
   
    protected void initAllData()
    {
    	initTwitterContentData();
    	initTwitterSessionContent();    
    	initTwitterUserData();
    }
    
    protected void restoreDefaultContent(Long defaultContentId)
    {
        twitterContent = new TwitterContent(defaultContentId,	
                DEFAULT_TITLE,
                DEFAULT_CONTENT,
                DEFAULT_ONLINE_INSTRUCTIONS,
                DEFAULT_OFFLINE_INSTRUCTIONS,
                DEFAULT_DEFINE_LATER,
                DEFAULT_FORCE_OFFLINE,
                DEFAULT_CONTENT_IN_USE,
				null,
				DEFAULT_CONTENT_IN_USE,
				TEST_CREATOR_USER_ID,
				TEST_DATE_CREATED,
				null);

        twitterDAO.saveTwitterContent(twitterContent);
 
    }
    
    //===========================
    // Helper Methods
    //===========================
     
    protected void assertTwitterSessionIsNull(Long id)
    {
        TwitterSession twitterSession = twitterSessionDAO.findTwitterSessionById(id);
   	   assertNull(twitterSession);
    }
    
    protected void assertTwitterContentIsNull(Long id)
    {
        TwitterContent twitterContent = twitterDAO.findTwitterContentById(id);
   	   assertNull(twitterContent);
    }
    
    protected void assertContentEqualsTestData(TwitterContent content)
    {
        	
    		assertEquals(content.getTitle(), TEST_TITLE);
    		assertEquals(content.getContent(), TEST_CONTENT);
    		assertEquals(content.getOnlineInstructions(), TEST_ONLINE_INSTRUCTIONS);
    		assertEquals(content.getOfflineInstructions(), TEST_OFFLINE_INSTRUCTIONS);
    		assertEquals(content.isDefineLater(), TEST_DEFINE_LATER);
    		assertEquals(content.isForceOffline(), TEST_FORCE_OFFLINE);
    		assertEquals(content.getCreatorUserId(), TEST_CREATOR_USER_ID);
    		assertEquals(content.getDateCreated(), TEST_DATE_CREATED);
    }
    
    protected void assertContentEqualsDefaultData(TwitterContent content)
    {
        	
    		assertEquals(content.getTitle(), DEFAULT_TITLE);
    		assertEquals(content.getContent(), DEFAULT_CONTENT);
    		assertEquals(content.getOnlineInstructions(), DEFAULT_ONLINE_INSTRUCTIONS);
    		assertEquals(content.getOfflineInstructions(), DEFAULT_OFFLINE_INSTRUCTIONS);
    		assertEquals(content.isDefineLater(), DEFAULT_DEFINE_LATER);
    		assertEquals(content.isForceOffline(), DEFAULT_FORCE_OFFLINE);
    		     } 
    
    protected void assertEqualsForSessionContent(TwitterSession ns)
    {
        assertEquals("Validate session id ",ns.getTwitterSessionId(), TEST_SESSION_ID);
		assertEquals("Validate content id ",ns.getTwitterContent().getTwitterContentId(), TEST_NB_ID);
		assertEquals("Validate session start date", ns.getSessionStartDate(), TEST_SESSION_START_DATE);
		assertEquals("Validate session end date", ns.getSessionEndDate(), TEST_SESSION_END_DATE);
		assertEquals("Validate session status", ns.getSessionStatus(), TEST_SESSION_STATUS);
    }
    
    protected void assertSessionObjectIsNull(Long sessionId)
    {
        TwitterSession nsObject = twitterSessionDAO.findTwitterSessionById(sessionId);
        assertNull(nsObject);
    }
    
    protected void assertEqualsForTwitterUser(TwitterUser user)
    {
        assertEquals("Validate user id",user.getUserId(), TEST_USER_ID);
		assertEquals("Validate username",user.getUsername(), TEST_USERNAME);
		assertEquals("Validate fullname", user.getFullname(), TEST_FULLNAME);
		assertEquals("Validate user status", user.getUserStatus(), TEST_USER_STATUS);	
		assertEquals("Validate session id",user.getTwitterSession().getTwitterSessionId(), TEST_SESSION_ID);
		
    }
    
    protected void assertEqualsForDefaultTwitterUser(TwitterUser user)
    {
        assertEquals("Validate user id",user.getUserId(), DEFAULT_USER_ID);
		assertEquals("Validate username",user.getUsername(), DEFAULT_USERNAME);
		assertEquals("Validate fullname", user.getFullname(), DEFAULT_FULLNAME);
		assertEquals("Validate user status", user.getUserStatus(), DEFAULT_USER_STATUS);	
		assertEquals("Validate session id",user.getTwitterSession().getTwitterSessionId(), DEFAULT_SESSION_ID);
		
    }
  
    protected void assertUserObjectIsNull(Long userId)
    {
        TwitterUser user = twitterUserDAO.getTwitterUser(userId, TEST_SESSION_ID);
        assertNull(user);
    }
    
    protected void assertAttachmentData(TwitterAttachment attachment)
	 {
	     assertEquals("Validating the filename:", attachment.getFilename(), TEST_FILENAME);
	     assertEquals("Validating whether it is an online file", attachment.isOnlineFile(), TEST_IS_ONLINE_FILE);
	     assertEquals("Validating the tool content id", attachment.getTwitterContent().getTwitterContentId(), TEST_NB_ID);
	     assertEquals("Validating the Uuid", attachment.getUuid(), TEST_UUID);
	 }
}
